#include "vpptest.h"
#include <cmath>
#include <tchar.h>

using namespace std;

#define MSDK_PRINT_RET_MSG(ERR) {_tprintf(_T("\nReturn on error: error code %d,\t%s\t%d\n\n"), ERR, _T(__FILE__), __LINE__);}

#define MSDK_CHECK_ERROR(P, X, ERR)              {if ((X) == (P)) { return ERR; }}
#define MSDK_CHECK_RESULT(P, X, ERR)             {if ((X) > (P)) {MSDK_PRINT_RET_MSG(ERR); return ERR;}}
#define MSDK_CHECK_POINTER(P, ...)               {if (!(P)) {return __VA_ARGS__;}}

#define MSDK_ZERO_MEMORY(VAR)                    {memset(&VAR, 0, sizeof(VAR));}
#define MSDK_MAX(A, B)                           (((A) > (B)) ? (A) : (B))
#define MSDK_MIN(A, B)                           (((A) < (B)) ? (A) : (B))
#define MSDK_ALIGN16(value)                      (((value + 15) >> 4) << 4) // round up to a multiple of 16
#define MSDK_ALIGN32(value)                      (((value + 31) >> 5) << 5) // round up to a multiple of 32
#define MSDK_ALIGN(value, alignment)             (alignment) * ( (value) / (alignment) + (((value) % (alignment)) ? 1 : 0))
#define MSDK_ARRAY_LEN(value)                    (sizeof(value) / sizeof(value[0]))

#define MSDK_MEMCPY_VAR(dstVarName, src, count) memcpy(&(dstVarName), (src), (count))

static void *inner_malloc(size_t size)
{
	void *ptr = NULL;
	long diff;

	/* let's disallow possible ambiguous cases */
	if (size > (INT32_MAX-32))
		return NULL;

	ptr = malloc(size+32);
	if(!ptr)
		return ptr;
	diff= ((-(long)ptr - 1)&(32-1)) + 1;
	ptr = (char*)ptr + diff;
	((char*)ptr)[-1]= diff;

	if(!ptr && !size)
		ptr= malloc(1);
	return ptr;
}
static void inner_free(void *ptr)
{
	long diff = ((char*)ptr)[-1];
	free((char*)ptr - diff);
}

mfxStatus ConvertFrameRate(mfxF64 dFrameRate, mfxU32* pnFrameRateExtN, mfxU32* pnFrameRateExtD)
{
	MSDK_CHECK_POINTER(pnFrameRateExtN, MFX_ERR_NULL_PTR);
	MSDK_CHECK_POINTER(pnFrameRateExtD, MFX_ERR_NULL_PTR);

	mfxU32 fr;

	fr = (mfxU32)(dFrameRate + .5);

	if (fabs(fr - dFrameRate) < 0.0001)
	{
		*pnFrameRateExtN = fr;
		*pnFrameRateExtD = 1;
		return MFX_ERR_NONE;
	}

	fr = (mfxU32)(dFrameRate * 1.001 + .5);

	if (fabs(fr * 1000 - dFrameRate * 1001) < 10)
	{
		*pnFrameRateExtN = fr * 1000;
		*pnFrameRateExtD = 1001;
		return MFX_ERR_NONE;
	}

	*pnFrameRateExtN = (mfxU32)(dFrameRate * 10000 + .5);
	*pnFrameRateExtD = 10000;

	return MFX_ERR_NONE;
}

////////////////////////////////////////////

int  SurfPool::alloc(const mfxFrameInfo& info, size_t size)
{
	mSurfArray.clear();
	mfxFrameSurface1 surf;
	memset(&surf, 0, sizeof(surf));
	memcpy(&surf.Info, &info, sizeof(mfxFrameInfo));
	mSurfArray.assign(size, surf);
	return 0;
}

void SurfPool::dealloc()
{
	mSurfArray.clear();
}

mfxFrameSurface1* SurfPool::attachFreeSurf(const uint8_t* ptr)
{
	mfxFrameSurface1* pSurf = NULL;
	for (int i = 0; i < mSurfArray.size(); i++) {
		if (0 == mSurfArray[i].Data.Locked) {
			pSurf = &mSurfArray[i];
		}
	}
	if (pSurf) {
		mfxU16 Width  = (mfxU16)MSDK_ALIGN32(pSurf->Info.Width);
		mfxU16 Height = (mfxU16)MSDK_ALIGN32(pSurf->Info.Height);
		pSurf->Data.Pitch = Width;
		pSurf->Data.Y     = (mfxU8*)ptr;
		pSurf->Data.U     = pSurf->Data.Y + Width * Height;
		pSurf->Data.V     = pSurf->Data.U + 1;
	}
	return pSurf;
}

////////////////////////////////////////////

VppProc::VppProc(const VideoStreamFormat& in, const VideoStreamFormat& out)
: mVpp(NULL)
{
	MSDK_ZERO_MEMORY(mVppParams);
	mVppParams.IOPattern = MFX_IOPATTERN_IN_SYSTEM_MEMORY | MFX_IOPATTERN_OUT_SYSTEM_MEMORY;

	/* in */ 
	mVppParams.vpp.In.FourCC       = MFX_FOURCC_NV12;
	mVppParams.vpp.In.ChromaFormat = MFX_CHROMAFORMAT_YUV420;
	mVppParams.vpp.In.PicStruct    = in.picStruct;
	ConvertFrameRate(in.frameRate, &mVppParams.vpp.In.FrameRateExtN, &mVppParams.vpp.In.FrameRateExtD);

	mVppParams.vpp.In.Width     = MSDK_ALIGN16(in.width);
	mVppParams.vpp.In.Height    = (MFX_PICSTRUCT_PROGRESSIVE == mVppParams.vpp.In.PicStruct)?
		MSDK_ALIGN16(in.height) : MSDK_ALIGN32(in.height);

	mVppParams.vpp.In.CropW = in.width;
	mVppParams.vpp.In.CropH = in.height;

	/* out */
	MSDK_MEMCPY_VAR(mVppParams.vpp.Out, &mVppParams.vpp.In, sizeof(mfxFrameInfo));
	mVppParams.vpp.Out.PicStruct = out.picStruct;

	mVppParams.vpp.Out.Width = MSDK_ALIGN16(out.width);
	mVppParams.vpp.Out.Height = (MFX_PICSTRUCT_PROGRESSIVE == mVppParams.vpp.Out.PicStruct)?
		MSDK_ALIGN16(out.height) : MSDK_ALIGN32(out.height);

	mVppParams.vpp.Out.CropW = out.width;
	mVppParams.vpp.Out.CropH = out.height;

	/* other */
	MSDK_ZERO_MEMORY(mVppDoNotUse);
	mVppDoNotUse.Header.BufferId = MFX_EXTBUFF_VPP_DONOTUSE;
	mVppDoNotUse.Header.BufferSz = sizeof(mVppDoNotUse);
	mVppDoNotUse.NumAlg          = 4;
	mVppDoNotUse.AlgList         = new mfxU32[mVppDoNotUse.NumAlg];    
	mVppDoNotUse.AlgList[0]      = MFX_EXTBUFF_VPP_DENOISE;
	mVppDoNotUse.AlgList[1]      = MFX_EXTBUFF_VPP_SCENE_ANALYSIS;
	mVppDoNotUse.AlgList[2]      = MFX_EXTBUFF_VPP_DETAIL;
	mVppDoNotUse.AlgList[3]      = MFX_EXTBUFF_VPP_PROCAMP;

	mVppExtParams.push_back((mfxExtBuffer *)&mVppDoNotUse);

	mVppParams.ExtParam = &mVppExtParams[0];
	mVppParams.NumExtParam = (mfxU16)mVppExtParams.size();
}

VppProc::~VppProc()
{
	close();
}

int  VppProc::open()
{
	mfxStatus sts = MFX_ERR_NONE;
	mfxIMPL impl = MFX_IMPL_HARDWARE2;
	mfxVersion version = { 0, 1 };
	sts = mSession.Init(impl, &version);
	if (sts != MFX_ERR_NONE)
	{
		impl = MFX_IMPL_HARDWARE2;
		sts = mSession.Init(impl, &version);
		if (sts != MFX_ERR_NONE)
		{
			printf("Error Code %d\n", sts);
			return -1;
		}
	}

	mVpp = new MFXVideoVPP(mSession);
	sts = mVpp->Init(&mVppParams);
	if (MFX_WRN_PARTIAL_ACCELERATION == sts) {
		printf("[WARNING] partial acceleration\n");
	}
	MSDK_CHECK_RESULT(sts, MFX_ERR_NONE, sts);

	mfxFrameAllocRequest VppRequest[2];

	MSDK_ZERO_MEMORY(VppRequest[0]);
	MSDK_ZERO_MEMORY(VppRequest[1]);
	sts = mVpp->QueryIOSurf(&mVppParams, VppRequest);
	MSDK_CHECK_RESULT(sts, MFX_ERR_NONE, sts);

	size_t iNum = VppRequest[0].NumFrameSuggested + 3;
	size_t oNum = VppRequest[1].NumFrameSuggested + 3;

	mSurfI.alloc(mVppParams.vpp.In, iNum);
	mSurfO.alloc(mVppParams.vpp.Out, oNum);

	return (int)sts;
}

void VppProc::close()
{
	if(mVpp != NULL) {
		mVpp->Close();
		delete mVpp;
		mVpp = NULL;
	}
	if(mVppDoNotUse.AlgList != NULL) {
		delete[] mVppDoNotUse.AlgList;
		mVppDoNotUse.AlgList = NULL;
	}
	mVppDoNotUse.NumAlg = 0;
	mVppExtParams.clear();
}


int  VppProc::process(const uint8_t* pIn, uint8_t* pOut)
{
	mfxStatus sts = MFX_ERR_NONE;
	mfxSyncPoint sp = NULL;

	mfxFrameSurface1* pSurfI = mSurfI.attachFreeSurf(pIn);
	mfxFrameSurface1* pSurfO = mSurfO.attachFreeSurf(pOut);

	assert(pSurfI && pSurfO);

	for (;;) {
		sts = mVpp->RunFrameVPPAsync(pSurfI, pSurfO, NULL, &sp);

		if (MFX_ERR_NONE < sts && !sp) {
			if (MFX_WRN_DEVICE_BUSY == sts)
				Sleep(1);
			continue;
		} else if (MFX_ERR_NONE < sts && sp) {
			sts = MFX_ERR_NONE;
		}
		break;
	}

	assert(sp && (sts == MFX_ERR_NONE));

	sts = mSession.SyncOperation(sp, MSDK_WAIT_INTERVAL);
	if (MFX_ERR_NONE == sts) {
		assert(pSurfO->Data.Locked == 0);
	}
	else
	{
		printf("VPP error code %d.\n", sts);
	}
	sp = NULL;
	return 0;
}


////////////////////////////////////////////

VppTestCase::VppTestCase(const char* file, const VideoStreamFormat& in, const VideoStreamFormat& out)
: mVpp(in, out)
, mFmtIn(in)
, mFmtOut(out)
{
	if (file) mYUVFile.open(file, ios::binary|ios::in);
}

int VppTestCase::exec()
{
	if (mVpp.open()) {
		printf("[Error] Cannot open vpp\n");
		return -1;
	}
	int iSize = MSDK_ALIGN32(mFmtIn.width)  * MSDK_ALIGN32(mFmtIn.height)  * 3 / 2;
	int oSize = MSDK_ALIGN32(mFmtOut.width) * MSDK_ALIGN32(mFmtOut.height) * 3 / 2;
	for (;;) {
		char* pIn  = (char*)inner_malloc(iSize);
		char* pOut = (char*)inner_malloc(oSize);
		if (mYUVFile.is_open()) {
			mYUVFile.read(pIn, iSize);
			if (mYUVFile.eof()) {
				mYUVFile.clear();
				mYUVFile.seekg(0, ios::beg);
				mYUVFile.read(pIn, iSize);
			}
		} else {
			memset(pIn, 0, iSize);
		}
		char* pIn2  = (char*)inner_malloc(iSize);
		memcpy(pIn2, pIn, iSize);
		mVpp.process((uint8_t*)pIn2, (uint8_t*)pOut);
		inner_free(pIn2);
		inner_free(pIn);
		inner_free(pOut);

		// 		TMediaSample* pSample = TMediaSample::createInstance();
		// 		pSample->setData(NULL, iSize);
		// 		if (mYUVFile.is_open()) {
		// 			mYUVFile.read(pSample->data(), iSize);
		// 			if (mYUVFile.eof()) {
		// 				mYUVFile.clear();
		// 				mYUVFile.seekg(0, ios::beg);
		// 				mYUVFile.read(pSample->data(), iSize);
		// 			}
		// 		} else {
		// 			memset(pSample->data(), 0, iSize);
		// 		}
		// 		TMediaSample* iSample = TMediaSample::createInstance(pSample, false);
		// 		TMediaSample* oSample = TMediaSample::createInstance(pSample, false);
		// 		mVpp.process((uint8_t*)iSample->data(), (uint8_t*)oSample->data());
		// 		iSample->release();
		// 		oSample->release();
		// 		pSample->release();

	}
}